/********************************************************************************************************************************************************
*  author: Michael Mandel (mmandel@apple.com)                                                                                                           *
*  created: 5/30/06                                                                                                                                     *
*  Implements the basic fixed function features of OpenGL we use in shaders - this is common code that may be shared between shaders.                   *
*  This relies on uniform parameters created by Coin and updated based on current openGL state, causing quite a few conditionals in the shader code...  *
*********************************************************************************************************************************************************/

vec4 Ambient;
vec4 Diffuse;
vec4 Specular;

uniform int coin_HasSeparateSpecular;
uniform int coin_NumEnabledLights;

void spotLight(in int i, in vec3 normal, in vec3 eye, in vec3 ecPosition3)
{
   float nDotVP;            // normal . light direction
   float nDotHV;            // normal . light half vector
   float pf;                // power factor
   float spotDot;           // cosine of angle between spotlight
   float spotAttenuation;   // spotlight attenuation factor
   float attenuation;       // computed attenuation factor
   float d;                 // distance from surface to light source
   vec3  VP;                // direction from surface to light position
   vec3  halfVector;        // direction of maximum highlights

   // Compute vector from surface to light position
   VP = vec3 (gl_LightSource[i].position) - ecPosition3;

   // Compute distance between surface and light position
   d = length(VP);

   // Normalize the vector from surface to light position
   VP = normalize(VP);

   // Compute attenuation
   attenuation = 1.0 / (gl_LightSource[i].constantAttenuation +
       gl_LightSource[i].linearAttenuation * d +
       gl_LightSource[i].quadraticAttenuation * d * d);

   // See if point on surface is inside cone of illumination
   spotDot = dot(-VP, normalize(gl_LightSource[i].spotDirection));

   if (spotDot < gl_LightSource[i].spotCosCutoff)
   {
       spotAttenuation = 0.0; // light adds no contribution
   }
   else
   {
       spotAttenuation = pow(spotDot, gl_LightSource[i].spotExponent);

   }
   // Combine the spotlight and distance attenuation.
   attenuation *= spotAttenuation;

   halfVector = normalize(VP + eye);

   nDotVP = max(0.0, dot(normal, VP));
   nDotHV = max(0.0, dot(normal, halfVector));

   if (nDotVP == 0.0)
   {
       pf = 0.0;
   }
   else
   {
       pf = pow(nDotHV, gl_FrontMaterial.shininess);

   }
   Ambient  += gl_LightSource[i].ambient * attenuation;
   Diffuse  += gl_LightSource[i].diffuse * nDotVP * attenuation;
   Specular += gl_LightSource[i].specular * pf * attenuation;

}

void directionalLight(in int i, in vec3 normal)
{
   float nDotVP;         // normal . light direction
   float nDotHV;         // normal . light half vector
   float pf;             // power factor

   nDotVP = max(0.0, dot(normal, normalize(vec3
(gl_LightSource[i].position))));
   nDotHV = max(0.0, dot(normal, vec3 (gl_LightSource[i].halfVector)));

   if (nDotVP == 0.0)
   {
       pf = 0.0;
   }
   else
   {
       pf = pow(nDotHV, gl_FrontMaterial.shininess);

   }
   Ambient  += gl_LightSource[i].ambient;
   Diffuse  += gl_LightSource[i].diffuse * nDotVP;
   Specular += gl_LightSource[i].specular * pf;
}

void pointLight(in int i, in vec3 normal, in vec3 eye, in vec3 ecPosition3)
{
   float nDotVP;       // normal . light direction
   float nDotHV;       // normal . light half vector
   float pf;           // power factor
   float attenuation;  // computed attenuation factor
   float d;            // distance from surface to light source
   vec3  VP;           // direction from surface to light position
   vec3  halfVector;   // direction of maximum highlights

   // Compute vector from surface to light position
   VP = vec3 (gl_LightSource[i].position) - ecPosition3;

   // Compute distance between surface and light position
   d = length(VP);

   // Normalize the vector from surface to light position
   VP = normalize(VP);

   // Compute attenuation
   attenuation = 1.0 / (gl_LightSource[i].constantAttenuation +
       gl_LightSource[i].linearAttenuation * d +
       gl_LightSource[i].quadraticAttenuation * d * d);

   halfVector = normalize(VP + eye);

   nDotVP = max(0.0, dot(normal, VP));
   nDotHV = max(0.0, dot(normal, halfVector));

   if (nDotVP == 0.0)
   {
       pf = 0.0;
   }
   else
   {
       pf = pow(nDotHV, gl_FrontMaterial.shininess);

   }
   Ambient  += gl_LightSource[i].ambient * attenuation;
   Diffuse  += gl_LightSource[i].diffuse * nDotVP * attenuation;
   Specular += gl_LightSource[i].specular * pf * attenuation;
}

vec3 fnormal(void)
{
    //Compute the normal 
    vec3 normal = gl_NormalMatrix * gl_Normal;
    normal = normalize(normal);
    return normal;
}

void fixed_funcComputeTexCoords(in vec3 normal, in vec4 ecPosition)
{
    gl_TexCoord[0] = gl_TextureMatrix[0]*gl_MultiTexCoord0;
}

void fixed_funcComputeVertexColor(in vec3 normal, in vec4 ecPosition)
{
    vec4 color;
    vec3 ecPosition3;
    vec3 eye;
    int i;

    ecPosition3 = (vec3 (ecPosition)) / ecPosition.w;
    eye = vec3 (0.0, 0.0, 1.0);

    // Clear the light intensity accumulators
    Ambient  = vec4 (0.0);
    Diffuse  = vec4 (0.0);
    Specular = vec4 (0.0);

//    pointLight(0, normal, eye, ecPosition3);
//    pointLight(1, normal, eye, ecPosition3);
//    pointLight(2, normal, eye, ecPosition3);
//    pointLight(3, normal, eye, ecPosition3);
    for(i = 0 ; i < coin_NumEnabledLights ; i++) {
        if(gl_LightSource[i].position.w == 0.0) {
            //directionalLight(i, normal);
            pointLight(i, normal, eye, ecPosition3);
        }
        else if(gl_LightSource[i].spotCutoff == 180.0) {
            pointLight(i, normal, eye, ecPosition3);
        }
        else {
            //spotLight(i, normal, eye, ecPosition3);
            pointLight(i, normal, eye, ecPosition3);
        }
    }

    color = gl_FrontLightModelProduct.sceneColor +
      Ambient  * gl_FrontMaterial.ambient +
      Diffuse  * gl_FrontMaterial.diffuse;
    
    if(coin_HasSeparateSpecular == 1) {
        gl_FrontSecondaryColor = Specular * gl_FrontMaterial.specular;
    }
    
    color = clamp( color, 0.0, 1.0 );
    gl_FrontColor = color;

}
